/*
   Axe clone
   Version 0.1
   Damian Yerrick (tepples@spamcop.net)
   http://www.pineight.com/ds/

   Subject to copyright license in README.txt

   Based on framebuffer_demo1 ARM9 Code 
   Chris Double (chris.double@double.co.nz)
   http://www.double.co.nz/nintendo_ds
*/

#include <nds.h>

//////////////////////////////////////////////////////////////////////


#define TOUCH_CAL_X1 (*(vs16*)0x027FFCD8)
#define TOUCH_CAL_Y1 (*(vs16*)0x027FFCDA)
#define TOUCH_CAL_X2 (*(vs16*)0x027FFCDE)
#define TOUCH_CAL_Y2 (*(vs16*)0x027FFCE0)
#define SCREEN_WIDTH    256
#define SCREEN_HEIGHT   192
s32 TOUCH_WIDTH;
s32 TOUCH_HEIGHT;
s32 TOUCH_OFFSET_X;
s32 TOUCH_OFFSET_Y;


//////////////////////////////////////////////////////////////////////


/* Frequencies of 32 different notes on an E major pentatonic scale */
const unsigned int axeFreqs[32] =
{
  (int)SOUND_FREQ(  34.65*8),
  (int)SOUND_FREQ(  41.20*8),
  (int)SOUND_FREQ(  46.25*8),
  (int)SOUND_FREQ(  51.91*8),
  (int)SOUND_FREQ(  61.73*8),

  (int)SOUND_FREQ(  69.30*8),
  (int)SOUND_FREQ(  82.41*8),
  (int)SOUND_FREQ(  92.50*8),
  (int)SOUND_FREQ( 103.82*8),
  (int)SOUND_FREQ( 123.47*8), 

  (int)SOUND_FREQ( 138.591*8),
  (int)SOUND_FREQ( 164.841*8),
  (int)SOUND_FREQ( 184.997*8),
  (int)SOUND_FREQ( 207.652*8),
  (int)SOUND_FREQ( 246.942*8),

  (int)SOUND_FREQ( 277.183*8),
  (int)SOUND_FREQ( 329.628*8),
  (int)SOUND_FREQ( 369.994*8),
  (int)SOUND_FREQ( 415.305*8),
  (int)SOUND_FREQ( 493.883*8),

  (int)SOUND_FREQ( 554.37*8),
  (int)SOUND_FREQ( 659.25*8),
  (int)SOUND_FREQ( 739.99*8),
  (int)SOUND_FREQ( 830.61*8),
  (int)SOUND_FREQ( 987.77*8),

  (int)SOUND_FREQ(1108.73*8),
  (int)SOUND_FREQ(1318.51*8),
  (int)SOUND_FREQ(1479.98*8),
  (int)SOUND_FREQ(1661.22*8),
  (int)SOUND_FREQ(1975.33*8),

  (int)SOUND_FREQ(2217.46*8),
  (int)SOUND_FREQ(2637.02*8)
};

int axePhase = 0;
int axeNextFreq = -1;
int axeLastPan = 0;
int axeLastDuty = 0;
int axeNextDuty = 0;

void axe(int touchXpx, int touchYpx) {
  int tempoCounter = IPC->heartbeat % 7;

  /*
   * Horizontal (X) = pitch and panning
   * Vertical (Y) = duty cycle
   */
  if(touchXpx > 255)
    touchXpx = 255;
  if(touchXpx >= 0) {
    axeNextFreq = touchXpx >> 3;
    axeNextDuty = (touchYpx / 48) << 24;
  } else {
    axeNextFreq = -1;
  }

  // If it's time to start a note, do so
  if(tempoCounter == 0) {
    if(axeNextFreq >= 0) {
      axePhase = 7;
      axeLastDuty = axeNextDuty;
      axeLastPan = SOUND_PAN(4 * axeNextFreq + 2);
      SCHANNEL_TIMER(10) = axeFreqs[axeNextFreq];
    }
  }

  // Update the note
  if(axePhase >= 0) {
    int vol = (axePhase * axePhase * 3) >> 1;
    SCHANNEL_CR(10) = SCHANNEL_ENABLE | SOUND_FORMAT_PSG | axeLastPan
                   | axeLastDuty | vol;
    --axePhase;
  }
}

/*
 * Plays a repeating drum track and bass track.
 */
void runSound(void) {
  static const unsigned char bassNotes[4] = {5, 6, 4, 5};
  unsigned int tempoCounter = IPC->heartbeat % (56 * 16);
  unsigned int bassNote = axeFreqs[(int)bassNotes[tempoCounter / (56 * 4)]];

  tempoCounter %= 56;


  if(tempoCounter >= 28 && tempoCounter < 36) {
    // play snare
    int snarePhase = ((43 - tempoCounter) * (43 - tempoCounter)) >> 3;

    SCHANNEL_CR(14) = SCHANNEL_ENABLE | SOUND_FORMAT_PSG | SOUND_PAN(72)
                   | snarePhase;
    SCHANNEL_TIMER(14) = SOUND_FREQ(10923);
  } else {
    // play hat
    int hatPhase = tempoCounter;
    if(hatPhase >= 36 && hatPhase < 42)
      hatPhase -= 36;
    if(hatPhase >= 28)
      hatPhase -= 28;
    if(hatPhase >= 14)
      hatPhase -= 14;
    else
      hatPhase *= 3;
    if(hatPhase >= 14)
      hatPhase = 0;
    else
      hatPhase = 14 - hatPhase;
    hatPhase = (hatPhase * hatPhase) >> 4;

    SCHANNEL_CR(14) = SCHANNEL_ENABLE | SOUND_FORMAT_PSG | SOUND_PAN(64)
                   | hatPhase;
    SCHANNEL_TIMER(14) = SOUND_FREQ(32768);
  }

  // play kick and bass
  {
    int kickPhase = tempoCounter;
    if(kickPhase >= 28)
      kickPhase -= 28;
    if(kickPhase >= 14) {
      // Play bass
      int kickVol = (28 - kickPhase) * 4;
      SCHANNEL_CR(9) = SCHANNEL_ENABLE | SOUND_FORMAT_PSG | SOUND_PAN(64)
                     | 0x03000000 | kickVol;
      SCHANNEL_TIMER(9) = bassNote;
    } else {
      // Play kick drum
      int kickVol = (kickPhase >= 8) ? 0 : 10 * (8 - kickPhase);
      SCHANNEL_CR(9) = SCHANNEL_ENABLE | SOUND_FORMAT_PSG | SOUND_PAN(64)
                     | 0x03000000 | kickVol;
      SCHANNEL_TIMER(9) = SOUND_FREQ(660 * 8) * (kickPhase + 4);
    }
  }
}


//////////////////////////////////////////////////////////////////////


void InterruptHandler(void) {
  static int heartbeat = 0;
  int interrupts = REG_IF;
 
  if (interrupts & IRQ_VBLANK) {
    uint16 but=0, batt=0, aux=0;
    int t1=0, t2=0;
    uint32 temp=0;
    uint8 ct[sizeof(IPC->curtime)];

    
    // Update the heartbeat
    heartbeat++;
 
    // Read the X/Y buttons and the /PENIRQ line
    but = REG_KEYXY;
    if (!(but & 0x40)) {
      touchPosition tp = touchReadXY();
      IPC->touchX   = tp.x;
      IPC->touchY   = tp.y;
      IPC->touchXpx = tp.px;
      IPC->touchYpx = tp.py;
      IPC->touchZ1  = tp.z1;
      IPC->touchZ2  = tp.z2;
    } else {
      // If pen is up, leave touch IPCs unmodified.
    }

    batt = touchRead(TSC_MEASURE_BATTERY);
    aux  = touchRead(TSC_MEASURE_AUX);

    // Read the time
    rtcGetTime((uint8 *)ct);
    BCDToInteger((uint8 *)&(ct[1]), 7);
 
    // Read the temperature
    temp = touchReadTemperature(&t1, &t2);
 
    // Update the IPC struct
    IPC->heartbeat = heartbeat;
    IPC->buttons   = but;
#if 0
    IPC->touchX    = x;
    IPC->touchY    = y;
    IPC->touchXpx  = xpx;
    IPC->touchYpx  = ypx;
    IPC->touchZ1   = z1;
    IPC->touchZ2   = z2;
#endif
    IPC->battery   = batt;
    IPC->aux       = aux;

    for(int i=0; i<sizeof(ct); i++) {
      IPC->curtime[i] = ct[i];
    }

    IPC->temperature = temp;
    IPC->tdiode1 = t1;
    IPC->tdiode2 = t2;


    //sound code  :)
    runSound();
    axe((but & 0x40) ? -1 : IPC->touchXpx, IPC->touchYpx);
  }
 
  // Acknowledge interrupts
  REG_IF = interrupts;
}
 

//////////////////////////////////////////////////////////////////////
 

int main(int argc, char ** argv) {
  // Reset the clock if needed
  rtcReset();

  // Compute touch parameters
  TOUCH_WIDTH  = TOUCH_CAL_X2 - TOUCH_CAL_X1;
  TOUCH_HEIGHT = TOUCH_CAL_Y2 - TOUCH_CAL_Y1;
  TOUCH_OFFSET_X = ( ((SCREEN_WIDTH -60) * TOUCH_CAL_X1) / TOUCH_WIDTH  ) - 28;
  TOUCH_OFFSET_Y = ( ((SCREEN_HEIGHT-60) * TOUCH_CAL_Y1) / TOUCH_HEIGHT ) - 28;

  //enable sound
  SOUND_CR = SOUND_ENABLE | SOUND_VOL(0x7F);
  IPC->soundData = 0;
  SCHANNEL_CR(10) = 0;
 
  // Set up the interrupt handler
  REG_IME = 0;
  IRQ_HANDLER = &InterruptHandler;
  REG_IE = IRQ_VBLANK;
  REG_IF = ~0;
  REG_DISPSTAT = DISP_VBLANK_IRQ;
  REG_IME = 1;

  // Keep the ARM7 out of main RAM
  while (1) swiWaitForVBlank();
  return 0;
}

 
//////////////////////////////////////////////////////////////////////

