/*
   Drag a box around!
   Version 0.1
   Damian Yerrick (tepples@spamcop.net)
   http://www.pineight.com/ds/

   Subject to copyright license in README.txt

   Based on framebuffer_demo1 ARM9 Code 
   Chris Double (chris.double@double.co.nz)
   http://www.double.co.nz/nintendo_ds
*/

#include <nds.h>
#include <sys/types.h>

static uint16 shape_color = RGB15(31, 0, 0) | BIT(15);

void rectfill(int x, int y, int width, int height, uint16* buffer, uint16 color)
{
  if(x < 0) {
    width += x;
    x = 0;
  }
  if(y < 0) {
    height += y;
    y = 0;
  }
  if(width > 256 - x) {
    width = 256 - x;
  }
  if(height > 192 - y) {
    height = 192 - y;
  }
  if(width > 0 && height > 0) {
    buffer += y * 256 + x;
    for(;
        height > 0;
        --height, buffer += 256) {
      uint16* line = buffer;
      for(int j = width; j > 0; --j) {
        *line++ = color;
      }
    }
  }
}

void isr(void) 
{
  int interrupts = REG_IF;

  VBLANK_INTR_WAIT_FLAGS |= interrupts;
  REG_IF = interrupts;
}

void setupISR(void)
{
  REG_IME = 0;
  IRQ_HANDLER = isr;
  REG_IE = IRQ_VBLANK;
  REG_IF = ~0;
  REG_DISPSTAT = DISP_VBLANK_IRQ;
  REG_IME = 1;
}

#define KEY_X (IPC_X << 16)
#define KEY_Y (IPC_Y << 16)
#define KEY_PEN (IPC_PEN_DOWN << 16)
#define VRAM_MAIN ((uint16 *)0x06000000)
#define VRAM_SUB ((uint16 *)0x06200000)

static void upcvt_4bit(void *dst, const u8 *src, size_t len)
{
  u32 *out = dst;

  for(; len > 0; len--)
  {
    u32 dst_bits = 0;
    u32 src_bits = *src++;
    u32 x;

    for(x = 0; x < 8; x++)
    {
      dst_bits <<= 4;
      dst_bits |= src_bits & 1;
      src_bits >>= 1;
    }
    *out++ = dst_bits;
  }
}

extern const u8 text_chr[];

void ntputs(int x, int y, const char *d) {
  u16 *dst = (u16 *)(BG_MAP_RAM(31)) + y * 32 + x;
  while (*d) {
    *dst++ = *d++ & 0xFF;
  }
}

const char blkTypes[16] = " IJLOSTZG*$$    ";
void drawBlock(int x, int y, int c) {
  c = blkTypes[(c & 0xF0) >> 4];
  if (x >= 0 && x < 10 && y >= 0 && y < 20) {
    u16 *dst = (u16 *)(BG_MAP_RAM(31)) + 32 * (21 - y) + x + 1;
    *dst = c;
  }
}

int main(void)
{
  int xTrail[32], yTrail[32];
  int shape_x = 0;
  int shape_y = 0;
  int shape_width = 10;
  int shape_height = 10;

  powerON(POWER_ALL);

  videoSetMode(MODE_5_2D
               | DISPLAY_BG0_ACTIVE);
  videoSetModeSub(MODE_5_2D
                  | DISPLAY_BG2_ACTIVE);
  BGCTRL[0] = BG_16_COLOR | BG_TILE_BASE(0) | BG_MAP_BASE(31);
  BGCTRL_SUB[2] = BG_BMP16_256x256;
  vramSetMainBanks(VRAM_A_MAIN_BG_0x6000000, VRAM_B_LCD, 
                   VRAM_C_SUB_BG_0x6200000, VRAM_D_LCD);
                   
  // Load font
  upcvt_4bit((u32 *)(BG_TILE_RAM(0)), text_chr, 2048);
  
  // Load map
  {
    u16 *dst = (u16 *)(BG_MAP_RAM(31));
    for (int i = 0; i < 1024; ++i) {
      dst[i] = 0;
    }
    ntputs(17, 1, "AXE 0.03");
    ntputs(17, 2, " 2006");
    ntputs(17, 3, "Damian Yerrick");
    ntputs(17, 16, "Make music");
    ntputs(17, 17, "by sliding");
    ntputs(17, 18, "the stylus");
    ntputs(17, 19, "around the");
    ntputs(17, 20, "Touch Screen.");
    ntputs(17, 21, "Across: pitch");
    ntputs(17, 22, "Up/Down: timbre");
    
    for (int y = 1; y <= 22; ++y) {
      u16 *row = dst + 32 * y;
      row[0] = 1;
      row[11] = 1;
    }
    for (int x = 0; x < 10; ++x) {
      dst[32 * 1 + 1 + x] = '_';
      dst[32 * 22 + 1 + x] = 1;
    }
    
    drawBlock(0, 0, 0x60);
    drawBlock(1, 0, 0x60);
    drawBlock(2, 0, 0x60);
    drawBlock(1, 1, 0x60);
    drawBlock(2, 1, 0x70);
    drawBlock(3, 1, 0x70);
    drawBlock(3, 0, 0x70);
    drawBlock(4, 0, 0x70);
    drawBlock(6, 2, 0x30);
    drawBlock(6, 1, 0x30);
    drawBlock(6, 0, 0x30);
    drawBlock(7, 0, 0x30);
    drawBlock(9, 2, 0x20);
    drawBlock(9, 1, 0x20);
    drawBlock(9, 0, 0x20);
    drawBlock(8, 0, 0x20);
  }
  
  // Load palette
  {
    BG_PALETTE[0] = RGB5(4, 2, 0);
    BG_PALETTE[1] = RGB5(31, 31, 31);
  }

  // Set scrolling
  BG_OFFSET[0].x = 0;
  BG_OFFSET[0].y = 0;

  SUB_BG2_XDX = 0x100;
  SUB_BG2_XDY = 0;
  SUB_BG2_YDX = 0;
  SUB_BG2_YDY = 0x100;
  SUB_BG2_CY = 0;
  SUB_BG2_CX = 0;

  lcdMainOnTop();

  // Initialize and enable interrupts
  setupISR();

  while(1) {
    int joy = ((~REG_KEYINPUT) & 0xFFFF) | ((~IPC->buttons) << 16);

    if((joy & KEY_UP) && (shape_y > 0))
      shape_y -= 1;
    if((joy & KEY_DOWN) && (shape_y < SCREEN_HEIGHT - shape_height))
      shape_y += 1;

    if((joy & KEY_LEFT) && (shape_x > 0))
      shape_x -= 1;
    if((joy & KEY_RIGHT) && (shape_x < SCREEN_WIDTH - shape_width))
      shape_x += 1;

#if 0
    if((joy & KEY_A) && (shape_width < 32)) {
      shape_width += 1;
      if(shape_x > SCREEN_WIDTH - shape_width)
        shape_x -= 1;
    }
    if((joy & KEY_Y) && (shape_width > 1)) {
      shape_width -= 1;
    }
    if((joy & KEY_B) && (shape_height < 32)) {
      shape_height += 1;
      if(shape_y > SCREEN_HEIGHT - shape_height)
        shape_y -= 1;
    }
    if((joy & KEY_X) && (shape_height > 1)) {
      shape_height -= 1;
    }
#endif

    if(joy & KEY_PEN) {
      shape_x = IPC->touchXpx - (shape_width >> 1);
      shape_y = IPC->touchYpx - (shape_height >> 1);

      if(shape_x < 0)
        shape_x = 0;
      if(shape_x > SCREEN_WIDTH - shape_width)
        shape_x = SCREEN_WIDTH - shape_width;
      if(shape_y < 0)
        shape_y = 0;
      if(shape_y > SCREEN_HEIGHT - shape_height)
        shape_y = SCREEN_HEIGHT - shape_height;
      shape_color = RGB15(31, 15, 15) | 0x8000;
    }

    swiWaitForVBlank();
    {
      int t = IPC->heartbeat & 0x001F;
      xTrail[t] = shape_x;
      yTrail[t] = shape_y;

      for(int i = 0; i < 32; ++i) {
        int x = xTrail[(i + t + 1) & 0x001F];
        int y = yTrail[(i + t + 1) & 0x001F];
        rectfill(x, y, shape_width, shape_height, VRAM_SUB,
                 RGB15(i, 0, 0) | 0x8000);
      }
    }
  }

  return 0;
}
